--- Welder Exhaust Effect Script
--- Author: Zyzex
--- Description: Adds working exhaust effect to the Miller Welder (FS25)
--- Modified: Uses custom exhaust.i3d with dynamic color change

WelderExhaust = {}

function WelderExhaust.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(Motorized, specializations)
end

function WelderExhaust.registerEventListeners(vehicleType)
    SpecializationUtil.registerEventListener(vehicleType, "onLoad", WelderExhaust)
    SpecializationUtil.registerEventListener(vehicleType, "onDelete", WelderExhaust)
    SpecializationUtil.registerEventListener(vehicleType, "onUpdate", WelderExhaust)
end

function WelderExhaust:onLoad(savegame)
    self.spec_welderExhaust = {}
    local spec = self.spec_welderExhaust
    
    spec.isExhaustActive = false
    spec.darkExhaustNode = nil
    spec.normalExhaustNode = nil
    spec.startupTimer = 0
    spec.isStartupPhase = false
    spec.transitionTimer = 0
    spec.isTransitioning = false
    
    -- Find the exhaust particle node from i3d mappings
    spec.exhaustNode = I3DUtil.indexToObject(self.components, "0>0|8")
    
    if spec.exhaustNode ~= nil then
        -- Load custom exhaust for BLACK smoke startup
        local effectFilename = "effects/exhaust.i3d"
        local i3dFilename = Utils.getFilename(effectFilename, self.baseDirectory)
        
        local rootNode1, sharedLoadRequestId1 = g_i3DManager:loadSharedI3DFile(i3dFilename, false, false)
        if rootNode1 ~= nil and rootNode1 ~= 0 then
            local exhaustNode = getChildAt(rootNode1, 0)
            if exhaustNode ~= nil and exhaustNode ~= 0 then
                spec.darkExhaustNode = clone(exhaustNode, false, false, true)
                link(spec.exhaustNode, spec.darkExhaustNode)
                
                -- Scale for thick black startup smoke
                setScale(spec.darkExhaustNode, 0.08, 0.08, 0.08)
                setVisibility(spec.darkExhaustNode, false)
                
                -- Set to MUCH DARKER BLACK color (almost black)
                if getHasShaderParameter(spec.darkExhaustNode, "exhaustColor") then
                    setShaderParameter(spec.darkExhaustNode, "exhaustColor", 0.02, 0.02, 0.02, 1.0, false)
                end
                
                -- Reduce particle lifespan to make them disappear closer
                if getHasShaderParameter(spec.darkExhaustNode, "lifespan") then
                    setShaderParameter(spec.darkExhaustNode, "lifespan", 0.3, 0, 0, 0, false)
                end
                
                -- Reduce speed to keep particles closer
                if getHasShaderParameter(spec.darkExhaustNode, "emitCountScale") then
                    setShaderParameter(spec.darkExhaustNode, "emitCountScale", 0.5, 0, 0, 0, false)
                end
            end
            delete(rootNode1)
        end
        
        -- Load custom exhaust for GRAY smoke normal running
        local rootNode2, sharedLoadRequestId2 = g_i3DManager:loadSharedI3DFile(i3dFilename, false, false)
        if rootNode2 ~= nil and rootNode2 ~= 0 then
            local exhaustNode = getChildAt(rootNode2, 0)
            if exhaustNode ~= nil and exhaustNode ~= 0 then
                spec.normalExhaustNode = clone(exhaustNode, false, false, true)
                link(spec.exhaustNode, spec.normalExhaustNode)
                
                -- Scale for light gray running smoke
                setScale(spec.normalExhaustNode, 0.05, 0.05, 0.05)
                setVisibility(spec.normalExhaustNode, false)
                
                -- Set to MUCH DARKER GRAY color (almost as dark as startup)
                if getHasShaderParameter(spec.normalExhaustNode, "exhaustColor") then
                    setShaderParameter(spec.normalExhaustNode, "exhaustColor", 0.15, 0.15, 0.15, 1.0, false)
                end
                
                -- Reduce particle lifespan to make them disappear closer
                if getHasShaderParameter(spec.normalExhaustNode, "lifespan") then
                    setShaderParameter(spec.normalExhaustNode, "lifespan", 0.4, 0, 0, 0, false)
                end
                
                -- Reduce speed to keep particles closer
                if getHasShaderParameter(spec.normalExhaustNode, "emitCountScale") then
                    setShaderParameter(spec.normalExhaustNode, "emitCountScale", 0.6, 0, 0, 0, false)
                end
            end
            delete(rootNode2)
        end
    end
end

function WelderExhaust:onDelete()
    local spec = self.spec_welderExhaust
    
    if spec.darkExhaustNode ~= nil then
        setVisibility(spec.darkExhaustNode, false)
        delete(spec.darkExhaustNode)
    end
    
    if spec.normalExhaustNode ~= nil then
        setVisibility(spec.normalExhaustNode, false)
        delete(spec.normalExhaustNode)
    end
end

function WelderExhaust:onUpdate(dt)
    local spec = self.spec_welderExhaust
    
    if spec.darkExhaustNode == nil or spec.normalExhaustNode == nil then
        return
    end
    
    local motorSpec = self.spec_motorized
    if motorSpec == nil then
        return
    end
    
    -- Check if motor is running by RPM
    local isMotorRunning = false
    if motorSpec.motor ~= nil then
        local rpm = motorSpec.motor.lastMotorRpm or 0
        isMotorRunning = rpm > 100
    end
    
    -- Handle startup dark smoke phase
    if spec.isStartupPhase then
        spec.startupTimer = spec.startupTimer + dt
        
        -- After 1000ms (1 second), start transition
        if spec.startupTimer >= 1000 and not spec.isTransitioning then
            spec.isTransitioning = true
            spec.transitionTimer = 0
            
            -- Show both for seamless transition
            setVisibility(spec.darkExhaustNode, true)
            setVisibility(spec.normalExhaustNode, true)
        end
    end
    
    -- Handle transition phase (fade between black and gray)
    if spec.isTransitioning then
        spec.transitionTimer = spec.transitionTimer + dt
        
        -- Transition over 1000ms (1 second) - same length as first phase
        if spec.transitionTimer >= 1000 then
            spec.isStartupPhase = false
            spec.isTransitioning = false
            spec.startupTimer = 0
            spec.transitionTimer = 0
            
            -- Hide black, keep gray
            setVisibility(spec.darkExhaustNode, false)
            setVisibility(spec.normalExhaustNode, true)
        end
    end
    
    -- Control visibility based on motor running
    if isMotorRunning then
        if not spec.isExhaustActive then
            spec.isExhaustActive = true
            
            -- Start with BLACK smoke
            spec.isStartupPhase = true
            spec.startupTimer = 0
            
            setVisibility(spec.darkExhaustNode, true)
            setVisibility(spec.normalExhaustNode, false)
        end
    else
        if spec.isExhaustActive then
            spec.isExhaustActive = false
            spec.isStartupPhase = false
            spec.isTransitioning = false
            spec.startupTimer = 0
            spec.transitionTimer = 0
            
            -- Hide both
            setVisibility(spec.darkExhaustNode, false)
            setVisibility(spec.normalExhaustNode, false)
        end
    end
end